<?php

declare(strict_types = 1);

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace TYPO3\CMS\Install\SystemEnvironment\ServerResponse;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\TransferException;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Core\Crypto\Random;
use TYPO3\CMS\Core\Http\Uri;
use TYPO3\CMS\Core\Messaging\FlashMessage;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Install\Controller\ServerResponseCheckController;
use TYPO3\CMS\Install\Status\AbstractStatus;
use TYPO3\CMS\Install\Status\ErrorStatus;
use TYPO3\CMS\Install\Status\StatusUtility;
use TYPO3\CMS\Install\SystemEnvironment\CheckInterface;
use TYPO3\CMS\Reports\Status;

/**
 * Checks how use web server is interpreting static files concerning
 * their `content-type` and evaluated content in HTTP responses.
 *
 * @internal should only be used from within TYPO3 Core
 */
class ServerResponseCheck implements CheckInterface
{
    const WRAP_FLAT = 1;
    const WRAP_NESTED = 2;

    /**
     * @var bool
     */
    protected $useMarkup;

    public function __construct(bool $useMarkup = true)
    {
        $this->useMarkup = $useMarkup;
    }

    public function asStatus(): Status
    {
        $statusUtility = GeneralUtility::makeInstance(StatusUtility::class);
        $statuses = $this->getStatus();
        $messages = array_map(static function (AbstractStatus $status) {
            return $status->getMessage();
        }, $statuses);
        $detailsLink = sprintf(
            '<p><a href="%s" rel="noreferrer" target="_blank">%s</a></p>',
            'https://docs.typo3.org/c/typo3/cms-core/master/en-us/Changelog/9.5.x/Feature-91354-IntegrateServerResponseSecurityChecks.html',
            'Please see documentation for further details...'
        );
        if ($statusUtility->filterBySeverity($statuses, 'error') !== []) {
            $title = 'Potential vulnerabilities';
            $label = $detailsLink;
            $severity = Status::ERROR;
        } elseif ($statusUtility->filterBySeverity($statuses, 'warning') !== []) {
            $title = 'Warnings';
            $label = $detailsLink;
            $severity = Status::WARNING;
        }
        return new Status(
            'Server Response',
            $title ?? 'OK',
            $this->wrapList($messages, $label ?? '', self::WRAP_NESTED),
            $severity ?? Status::OK
        );
    }

    /**
     * @return Status[]
     */
    public function getStatus(): array
    {
        $statuses = [];
        if (PHP_SAPI === 'cli-server') {
            $statuses['php_sapi_cli'] = GeneralUtility::makeInstance(
                Status::class,
                'Checks skipped',
                '',
                'Skipped for PHP_SAPI=cli-server',
                FlashMessage::WARNING
            );
            return $statuses;
        }
        $hostServerCheck = $this->processHostCheck();
        if ($hostServerCheck !== null) {
            $statuses['host_server_check'] = $hostServerCheck;
        }
        return $statuses;
    }

    /**
     * @return Status|null
     */
    protected function processHostCheck()
    {
        $random = GeneralUtility::makeInstance(Random::class);
        $randomHost = $random->generateRandomHexString(10) . '.random.example.org';
        $time = (string)time();
        $url = GeneralUtility::makeInstance(UriBuilder::class)->buildUriFromRoute(
            'install.server-response-check.host',
            ['src-time' => $time, 'src-hash' => ServerResponseCheckController::hmac($time)],
            UriBuilder::ABSOLUTE_URL
        );
        try {
            $client = new Client(['timeout' => 10]);
            $response = $client->request('GET', (string)$url, [
                'headers' => ['Host' => $randomHost],
                'allow_redirects' => false,
                'verify' => false,
            ]);
        } catch (TransferException $exception) {
            // it is expected that the previous request fails
            return null;
        }
        // in case we end up here, the server processed an HTTP request with invalid HTTP host header
        $messageParts = [];
        $locationHeader = $response->getHeaderLine('location');
        if (!empty($locationHeader) && (new Uri($locationHeader))->getHost() === $randomHost) {
            $messageParts[] = sprintf('HTTP Location header contained unexpected "%s"', $randomHost);
        }
        $data = json_decode((string)$response->getBody(), true);
        $serverHttpHost = $data['server.HTTP_HOST'] ?? null;
        $serverServerName = $data['server.SERVER_NAME'] ?? null;
        if ($serverHttpHost === $randomHost) {
            $messageParts[] = sprintf('HTTP_HOST contained unexpected "%s"', $randomHost);
        }
        if ($serverServerName === $randomHost) {
            $messageParts[] = sprintf('SERVER_NAME contained unexpected "%s"', $randomHost);
        }
        if ($messageParts !== []) {
            $status = GeneralUtility::makeInstance(ErrorStatus::class);
            $status->setTitle('Unexpected server response');
            $status->setMessage($this->wrapList($messageParts, (string)$url, self::WRAP_FLAT));
            return $status;
        }

        return null;
    }

    protected function wrapList(array $items, string $label, int $style): string
    {
        if (!$this->useMarkup) {
            return sprintf(
                '%s%s',
                $label ? $label . ': ' : '',
                implode(', ', $items)
            );
        }
        if ($style === self::WRAP_NESTED) {
            return sprintf(
                '%s<ul>%s</ul>',
                $label,
                implode('', $this->wrapItems($items, '<li>', '</li>'))
            );
        }
        return sprintf(
            '<p>%s%s</p>',
            $label,
            implode('', $this->wrapItems($items, '<br>', ''))
        );
    }

    protected function wrapItems(array $items, string $before, string $after): array
    {
        return array_map(
            function (string $item) use ($before, $after): string {
                return $before . $item . $after;
            },
            array_filter($items)
        );
    }
}
