<?php

declare(strict_types = 1);

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace TYPO3\CMS\Install\Updates;

use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Resource\Filter\FileExtensionFilter;
use TYPO3\CMS\Core\Resource\ResourceFactory;
use TYPO3\CMS\Core\Resource\Security\SvgSanitizer;
use TYPO3\CMS\Core\Utility\GeneralUtility;

class SvgFilesSanitization extends AbstractUpdate
{
    /**
     * @var string
     */
    protected $title = 'Sanitize existing SVG files in fileadmin folder';

    /**
     * Checks if an update is needed
     *
     * @param string &$description The description for the update
     * @return bool Whether an update is needed (TRUE) or not (FALSE)
     */
    public function checkForUpdate(&$description)
    {
        if ($this->isWizardDone()) {
            return false;
        }
        return true;
    }

    /**
     * Second step: Ask user to sanitize existing SVG files
     *
     * @param string $inputPrefix input prefix, all names of form fields have to start with this. Append custom name in [ ... ]
     * @return string HTML output
     */
    public function getUserInput($inputPrefix)
    {
        $markup = [];
        $markup[] = '<div class="panel panel-danger">';
        $markup[] = '  <div class="panel-heading">Are you really sure?</div>';
        $markup[] = '  <div class="panel-body">';
        $markup[] = '    <p>This upgrade wizard will sanitize all SVG files located in local file storages.</p>';
        $markup[] = '    <p>It is very likely that file contents will be changed.</p>';
        $markup[] = '    <p>Before continuing, please ensure a proper backup of *.svg and *.svgz files is in place before continuing.</p>';
        $markup[] = '    <p>Are you really sure, you want to do this now?</p>';
        $markup[] = '    <div class="btn-group clearfix" data-toggle="buttons">';
        $markup[] = '      <label class="btn btn-default active">';
        $markup[] = '        <input type="radio" name="' . $inputPrefix . '[sanitize]" value="0" checked="checked" /> no, don\'t sanitize';
        $markup[] = '      </label>';
        $markup[] = '      <label class="btn btn-default">';
        $markup[] = '        <input type="radio" name="' . $inputPrefix . '[sanitize]" value="1" /> yes, please sanitize';
        $markup[] = '      </label>';
        $markup[] = '    </div>';
        $markup[] = '  </div>';
        $markup[] = '</div>';
        return implode(LF, $markup);
    }

    /**
     * Performs the update
     *
     * @param array &$databaseQueries Queries done in this update
     * @param string &$customMessage Custom message
     *
     * @return bool
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    public function performUpdate(array &$databaseQueries, &$customMessage)
    {
        $requestParams = GeneralUtility::_GP('install');
        if (!isset($requestParams['values']['svgFilesSanitization']['sanitize'])) {
            return false;
        }
        $sanitize = (int)$requestParams['values']['svgFilesSanitization']['sanitize'];

        if ($sanitize === 1) {
            $this->executeUpdate();
        }

        $this->markWizardAsDone();
        return true;
    }

    private function executeUpdate(): bool
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getQueryBuilderForTable('sys_file_storage');
        $rows = $queryBuilder
            ->select('uid')
            ->from('sys_file_storage')
            ->where($queryBuilder->expr()->eq('is_writable', 1))
            ->execute()
            ->fetchAll();

        $resourceFactory = ResourceFactory::getInstance();
        foreach ($rows as $row) {
            $filter = GeneralUtility::makeInstance(FileExtensionFilter::class);
            $filter->setAllowedFileExtensions(['svg']);

            $storage = $resourceFactory->getStorageObject((int)$row['uid']);
            $storage->setFileAndFolderNameFilters([[$filter, 'filterFileList']]);
            $files = $storage->getFilesInFolder($storage->getRootLevelFolder(), 0, 0, true, true);

            $svgSanitizerService = GeneralUtility::makeInstance(SvgSanitizer::class);
            foreach ($files as $file) {
                $oldFileContent = $file->getContents();
                $newFileContent = $svgSanitizerService->sanitizeContent($oldFileContent);
                if ($oldFileContent !== $newFileContent) {
                    $file->setContents($newFileContent);
                }
            }
        }
        return true;
    }
}
