<?php
namespace TYPO3\CMS\Install\Controller\Action\Ajax;

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

use TYPO3\CMS\Core\Utility\GeneralUtility;

/**
 * Check if a younger version is available
 */
class CoreUpdateIsUpdateAvailable extends AbstractCoreUpdate
{
    /**
     * Executes the action
     *
     * @return string Rendered content
     */
    protected function executeAction()
    {
        $status = [];

        if ($this->coreVersionService->isInstalledVersionAReleasedVersion()) {
            $versionMaintenanceWindow = $this->coreVersionService->getMaintenanceWindow();
            $renderVersionInformation = false;

            if (!$versionMaintenanceWindow->isSupportedByCommunity() && !$versionMaintenanceWindow->isSupportedByElts()) {
                $status[] = $this->getMessage(
                    'error',
                    'Outdated version',
                    'The currently installed TYPO3 version ' . $this->coreVersionService->getInstalledVersion() . ' does not receive any further updates, please consider upgrading to a supported version!'
                );
                $action = null;
                $renderVersionInformation = true;
            } else {
                $currentVersion = $this->coreVersionService->getInstalledVersion();
                $isCurrentVersionElts = $this->coreVersionService->isCurrentInstalledVersionElts();
                $latestRelease = $this->coreVersionService->getYoungestPatchRelease();

                $availableReleases = [];
                if ($this->coreVersionService->isPatchReleaseSuitableForUpdate($latestRelease)) {
                    $availableReleases[] = $latestRelease;
                }
                if (!$versionMaintenanceWindow->isSupportedByCommunity()) {
                    if ($latestRelease->isElts()) {
                        // Check if there's a public release left that's not installed yet
                        $latestCommunityDrivenRelease = $this->coreVersionService->getYoungestCommunityPatchRelease();
                        if ($this->coreVersionService->isPatchReleaseSuitableForUpdate($latestCommunityDrivenRelease)) {
                            $availableReleases[] = $latestCommunityDrivenRelease;
                            $action = ['title' => 'Update now to version ' . $latestCommunityDrivenRelease->getVersion(), 'action' => 'updateRegular'];
                        }
                    } elseif (!$isCurrentVersionElts) {
                        // Inform user about ELTS being available soon if:
                        // - regular support ran out
                        // - the current installed version is no ELTS
                        // - no ELTS update was released, yet
                        $status[] = $this->getMessage(
                            'warning',
                            'ELTS will be available soon',
                            sprintf('The currently installed TYPO3 version %s doesn\'t receive any community-driven updates anymore, consider subscribing to Extended Long Term Support (ELTS) releases. Please read the information below.', $currentVersion)
                        );
                        $renderVersionInformation = true;
                    }
                }

                if ($availableReleases === []) {
                    $status[] = $this->getMessage(
                        'notice',
                        'Up to date',
                        'There are no TYPO3 updates available.'
                    );
                } else {
                    foreach ($availableReleases as $availableRelease) {
                        $isUpdateSecurityRelevant = $this->coreVersionService->isUpdateSecurityRelevant($availableRelease);
                        $versionString = $availableRelease->getVersion();
                        if ($availableRelease->isElts()) {
                            $versionString .= ' ELTS';
                        }

                        if ($isUpdateSecurityRelevant) {
                            $title = ($availableRelease->isElts() ? 'ELTS ' : '') . 'Security update available!';
                            $message = sprintf('The currently installed version is %s, update to security relevant released version %s is available.', $currentVersion, $versionString);
                            $severity = 'error';
                        } else {
                            $title = ($availableRelease->isElts() ? 'ELTS ' : '') . 'Update available!';
                            $message = sprintf('Currently installed version is %s, update to regular released version %s is available.', $currentVersion, $versionString);
                            $severity = 'warning';
                        }

                        if ($availableRelease->isElts()) {
                            if ($isCurrentVersionElts) {
                                $message .= ' Please visit my.typo3.org to download the release in your ELTS area.';
                            } else {
                                $message .= ' ' . sprintf('The currently installed TYPO3 version %s doesn\'t receive any community-driven updates anymore, consider subscribing to Extended Long Term Support (ELTS) releases. Please read the information below.', $currentVersion);
                            }

                            $renderVersionInformation = true;
                        }

                        $status[] = $this->getMessage($severity, $title, $message);
                    }
                }
            }

            if ($renderVersionInformation) {
                $supportedMajorReleases = $this->coreVersionService->getSupportedMajorReleases();
                $supportMessages = [];
                if (!empty($supportedMajorReleases['community'])) {
                    $supportMessages[] = sprintf('Currently community-supported TYPO3 versions: %s (more information at https://get.typo3.org).', implode(', ', $supportedMajorReleases['community']));
                }
                if (!empty($supportedMajorReleases['elts'])) {
                    $supportMessages[] = sprintf('Currently supported TYPO3 ELTS versions: %s (more information at https://typo3.com/elts).', implode(', ', $supportedMajorReleases['elts']));
                }
                $status[] = $this->getMessage(
                    'info',
                    'TYPO3 Version information',
                    implode(' ', $supportMessages)
                );
            }
        } else {
            $status = $this->getMessage('warning', 'Current version is a development version and can not be updated');
        }

        $this->view->assign('success', true);
        $this->view->assign('status', $status);
        if (isset($action)) {
            $this->view->assign('action', $action);
        }

        return $this->view->render();
    }

    /**
     * @param string $severity
     * @param string $title
     * @param string $message
     * @return \TYPO3\CMS\Install\Status\StatusInterface
     */
    protected function getMessage($severity, $title, $message = '')
    {
        /** @var $statusMessage \TYPO3\CMS\Install\Status\StatusInterface */
        $statusMessage = GeneralUtility::makeInstance('TYPO3\\CMS\\Install\\Status\\' . ucfirst($severity) . 'Status');
        $statusMessage->setTitle($title);
        $statusMessage->setMessage($message);

        return $statusMessage;
    }

    /**
     * @param string $title
     * @param string $action
     * @return array
     */
    protected function getAction($title, $action)
    {
        return [
            'title' => $title,
            'action' => $action,
        ];
    }
}
