<?php
namespace TYPO3\CMS\Install\Report;

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

use TYPO3\CMS\Backend\Utility\BackendUtility;
use TYPO3\CMS\Core\Registry;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Install\Service\Exception;
use TYPO3\CMS\Lang\LanguageService;
use TYPO3\CMS\Reports\Status;

/**
 * Provides an installation status report.
 */
class InstallStatusReport implements \TYPO3\CMS\Reports\StatusProviderInterface
{
    const WRAP_FLAT = 1;
    const WRAP_NESTED = 2;

    protected $useMarkup;

    /**
     * @var string
     */
    protected $reportList = 'FileSystem,RemainingUpdates,NewVersion';

    public function __construct(bool $useMarkup = true)
    {
        $this->useMarkup = $useMarkup;
    }

    /**
     * Compiles a collection of system status checks as a status report.
     *
     * @return Status[]
     */
    public function getStatus()
    {
        $reports = [];
        $reportMethods = explode(',', $this->reportList);
        foreach ($reportMethods as $reportMethod) {
            $reports[$reportMethod] = $this->{'get' . $reportMethod . 'Status'}();
        }
        return $reports;
    }

    /**
     * Checks for several directories being writable.
     *
     * @return Status Indicates status of the file system
     */
    protected function getFileSystemStatus()
    {
        $languageService = $this->getLanguageService();
        $value = $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_writable');
        $message = '';
        $severity = Status::OK;
        // Requirement level
        // -1 = not required, but if it exists may be writable or not
        //  0 = not required, if it exists the dir should be writable
        //  1 = required, doesn't have to be writable
        //  2 = required, has to be writable
        $checkWritable = [
            'typo3temp/' => 2,
            'typo3temp/assets/' => 2,
            'typo3temp/assets/compressed/' => 2,
            // only needed when GraphicalFunctions is used
            'typo3temp/assets/images/' => 0,
            // used in PageGenerator (inlineStyle2Temp) and Backend + Language JS files
            'typo3temp/assets/css/' => 2,
            'typo3temp/assets/js/' => 2,
            // fallback storage of FAL
            'typo3temp/assets/_processed_/' => 0,
            'typo3temp/var/' => 2,
            'typo3temp/var/transient/' => 2,
            'typo3temp/var/charset/' => 2,
            'typo3temp/var/locks/' => 2,
            'typo3conf/' => 2,
            'typo3conf/ext/' => 0,
            'typo3conf/l10n/' => 0,
            'uploads/' => 2,
            'uploads/media/' => 0,
            $GLOBALS['TYPO3_CONF_VARS']['BE']['fileadminDir'] => -1,
            $GLOBALS['TYPO3_CONF_VARS']['BE']['fileadminDir'] . '_temp_/' => 0,
        ];

        if ($GLOBALS['TYPO3_CONF_VARS']['EXT']['allowGlobalInstall']) {
            $checkWritable[TYPO3_mainDir . 'ext/'] = -1;
        }

        foreach ($checkWritable as $relPath => $requirementLevel) {
            if (!@is_dir(PATH_site . $relPath)) {
                // If the directory is missing, try to create it
                GeneralUtility::mkdir(PATH_site . $relPath);
            }
            if (!@is_dir(PATH_site . $relPath)) {
                if ($requirementLevel > 0) {
                    // directory is required
                    $value = $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_missingDirectory');
                    $message .= sprintf($languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_directoryDoesNotExistCouldNotCreate'), $relPath) . '<br />';
                    $severity = Status::ERROR;
                } else {
                    $message .= sprintf($languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_directoryDoesNotExist'), $relPath);
                    if ($requirementLevel == 0) {
                        $message .= ' ' . $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_directoryShouldAlsoBeWritable');
                    }
                    $message .= '<br />';
                    if ($severity < Status::WARNING) {
                        $value = $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_nonExistingDirectory');
                        $severity = Status::WARNING;
                    }
                }
            } else {
                if (!is_writable((PATH_site . $relPath))) {
                    switch ($requirementLevel) {
                        case 0:
                            $message .= sprintf($languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_directoryShouldBeWritable'), (PATH_site . $relPath)) . '<br />';
                            if ($severity < Status::WARNING) {
                                $value = $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_recommendedWritableDirectory');
                                $severity = Status::WARNING;
                            }
                            break;
                        case 2:
                            $value = $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_requiredWritableDirectory');
                            $message .= sprintf($languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_directoryMustBeWritable'), (PATH_site . $relPath)) . '<br />';
                            $severity = Status::ERROR;
                            break;
                        default:
                    }
                }
            }
        }
        return GeneralUtility::makeInstance(Status::class, $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_fileSystem'), $value, $message, $severity);
    }

    /**
     * Checks if there are still updates to perform
     *
     * @return Status Represents whether the installation is completely updated yet
     */
    protected function getRemainingUpdatesStatus()
    {
        $languageService = $this->getLanguageService();
        $value = $languageService->getLL('status_updateComplete');
        $message = '';
        $severity = Status::OK;

        // check if there are update wizards left to perform
        if (is_array($GLOBALS['TYPO3_CONF_VARS']['SC_OPTIONS']['ext/install']['update'])) {
            $versionAsInt = \TYPO3\CMS\Core\Utility\VersionNumberUtility::convertVersionNumberToInteger(TYPO3_version);
            $registry = GeneralUtility::makeInstance(Registry::class);
            foreach ($GLOBALS['TYPO3_CONF_VARS']['SC_OPTIONS']['ext/install']['update'] as $identifier => $className) {
                $updateObject = GeneralUtility::makeInstance($className, $identifier, $versionAsInt, null, $this);
                $markedDoneInRegistry = $registry->get('installUpdate', $className, false);
                if (!$markedDoneInRegistry && $updateObject->shouldRenderWizard()) {
                    // at least one wizard was found
                    $value = $languageService->getLL('status_updateIncomplete');
                    $severity = Status::WARNING;
                    $url = BackendUtility::getModuleUrl('system_extinstall');
                    $message = sprintf($languageService->sL('LLL:EXT:lang/Resources/Private/Language/locallang_core.xlf:warning.install_update'), '<a href="' . htmlspecialchars($url) . '">', '</a>');
                    break;
                }
            }
        }

        return GeneralUtility::makeInstance(Status::class, $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_remainingUpdates'), $value, $message, $severity);
    }

    /**
     * Checks if there is a new minor TYPO3 version to update to.
     *
     * @return Status Represents whether there is a new version available online
     */
    protected function getNewVersionStatus()
    {
        $languageService = $this->getLanguageService();
        /** @var \TYPO3\CMS\Install\Service\CoreVersionService $coreVersionService */
        $coreVersionService = GeneralUtility::makeInstance(\TYPO3\CMS\Install\Service\CoreVersionService::class);

        // No updates for development versions
        if (!$coreVersionService->isInstalledVersionAReleasedVersion()) {
            return GeneralUtility::makeInstance(Status::class, 'TYPO3', TYPO3_version, $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_isDevelopmentVersion'), Status::NOTICE);
        }

        // If fetching version matrix fails we can not do anything except print out the current version
        try {
            $versionMaintenanceWindow = $coreVersionService->getMaintenanceWindow();
        } catch (Exception\RemoteFetchException $remoteFetchException) {
            return GeneralUtility::makeInstance(Status::class, 'TYPO3', TYPO3_version, $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_remoteFetchException'), Status::NOTICE);
        }

        if (!$versionMaintenanceWindow->isSupportedByCommunity() && !$versionMaintenanceWindow->isSupportedByElts()) {
            // Version is not maintained
            $message = $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_versionOutdated');
            $status = Status::ERROR;
        } else {
            // There is an update available
            $availableReleases = [];
            $latestRelease = $coreVersionService->getYoungestPatchRelease();
            $isCurrentVersionElts = $coreVersionService->isCurrentInstalledVersionElts();

            if ($coreVersionService->isPatchReleaseSuitableForUpdate($latestRelease)) {
                $availableReleases[] = $latestRelease;
            }

            if (!$versionMaintenanceWindow->isSupportedByCommunity()) {
                if ($latestRelease->isElts()) {
                    $latestCommunityDrivenRelease = $coreVersionService->getYoungestCommunityPatchRelease();
                    if ($coreVersionService->isPatchReleaseSuitableForUpdate($latestCommunityDrivenRelease)) {
                        $availableReleases[] = $latestCommunityDrivenRelease;
                    }
                }
            }

            if ($availableReleases === []) {
                // Everything is fine, working with the latest version
                $message = '';
                $status = Status::OK;
            } else {
                $messages = [];
                $status = Status::WARNING;
                foreach ($availableReleases as $availableRelease) {
                    $versionString = $availableRelease->getVersion();
                    if ($availableRelease->isElts()) {
                        $versionString .= ' ELTS';
                    }
                    if ($coreVersionService->isUpdateSecurityRelevant($availableRelease)) {
                        $status = Status::ERROR;
                        $updateMessage = sprintf($languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_newVersionSecurityRelevant'), $versionString);
                    } else {
                        $updateMessage = sprintf($languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_newVersion'), $versionString);
                    }

                    if ($availableRelease->isElts()) {
                        if ($isCurrentVersionElts) {
                            $updateMessage .= ' ' . sprintf(
                                $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_elts_download'),
                                '<a href="https://my.typo3.org" target="_blank" rel="noopener">my.typo3.org</a>'
                            );
                        } else {
                            $updateMessage .= ' ' . sprintf(
                                $languageService->sL('LLL:EXT:install/Resources/Private/Language/Report/locallang.xlf:status_elts_subscribe'),
                                $coreVersionService->getInstalledVersion(),
                                '<a href="https://typo3.com/elts" target="_blank" rel="noopener">https://typo3.com/elts</a>'
                            );
                        }
                    }
                    $messages[] = $updateMessage;
                }
                $message = $this->wrapList($messages, count($messages) > 1 ? self::WRAP_NESTED : self::WRAP_FLAT);
            }
        }

        return GeneralUtility::makeInstance(Status::class, 'TYPO3', $coreVersionService->getInstalledVersion(), $message, $status);
    }

    protected function wrapList(array $items, int $style): string
    {
        if (!$this->useMarkup) {
            return implode(', ', $items);
        }
        if ($style === self::WRAP_NESTED) {
            return sprintf(
                '<ul>%s</ul>',
                implode('', $this->wrapItems($items, '<li>', '</li>'))
            );
        }
        return sprintf(
            '<p>%s</p>',
            implode('', $this->wrapItems($items, '<br>', ''))
        );
    }

    protected function wrapItems(array $items, string $before, string $after): array
    {
        return array_map(
            static function (string $item) use ($before, $after): string {
                return $before . $item . $after;
            },
            array_filter($items)
        );
    }

    /**
     * @return LanguageService
     */
    protected function getLanguageService()
    {
        return $GLOBALS['LANG'];
    }
}
