<?php
namespace TYPO3\CMS\Extbase\Tests\Unit\Reflection;

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

use Prophecy\Argument;
use TYPO3\CMS\Core\Cache\CacheManager;
use TYPO3\CMS\Core\Cache\Frontend\FrontendInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Object\ObjectManager;
use TYPO3\CMS\Extbase\Reflection\ClassSchema;
use TYPO3\CMS\Extbase\Reflection\ReflectionService;

/**
 * Test case
 * @firsttest test for reflection
 * @anothertest second test for reflection
 * @anothertest second test for reflection with second value
 */
class ReflectionServiceTest extends \TYPO3\TestingFramework\Core\Unit\UnitTestCase
{
    /**
     * @var array
     */
    protected $singletons;

    protected function setUp()
    {
        parent::setUp();
        $this->singletons = GeneralUtility::getSingletonInstances();
    }

    /**
     * @param array $foo The foo parameter
     * @return string
     */
    public function fixtureMethodForMethodTagsValues(array $foo)
    {
    }

    /**
     * @param bool $dummy
     * @param int $foo
     */
    public function fixtureMethodForMethodTagsValuesWithShortTypes($dummy, $foo)
    {
    }

    /**
     * @test
     */
    public function getClassTagsValues()
    {
        $service = new ReflectionService();
        $classValues = $service->getClassTagsValues(get_class($this));
        $this->assertEquals([
            'firsttest' => ['test for reflection'],
            'anothertest' => ['second test for reflection', 'second test for reflection with second value']
        ], $classValues);
    }

    /**
     * @test
     */
    public function getClassTagValues()
    {
        $service = new ReflectionService();
        $classValues = $service->getClassTagValues(get_class($this), 'firsttest');
        $this->assertEquals([
            'test for reflection',
        ], $classValues);
    }

    /**
     * @test
     */
    public function hasMethod()
    {
        $service = new ReflectionService();
        $this->assertTrue($service->hasMethod(get_class($this), 'fixtureMethodForMethodTagsValues'));
        $this->assertFalse($service->hasMethod(get_class($this), 'notExistentMethod'));
    }

    /**
     * @test
     */
    public function getMethodTagsValues()
    {
        $service = new ReflectionService();
        $tagsValues = $service->getMethodTagsValues(get_class($this), 'fixtureMethodForMethodTagsValues');
        $this->assertEquals([
            'param' => ['array $foo The foo parameter'],
            'return' => ['string']
        ], $tagsValues);
    }

    /**
     * @test
     */
    public function getMethodParameters()
    {
        $service = new ReflectionService();
        $parameters = $service->getMethodParameters(get_class($this), 'fixtureMethodForMethodTagsValues');
        $this->assertSame([
            'foo' => [
                'position' => 0,
                'byReference' => false,
                'array' => true,
                'optional' => false,
                'allowsNull' => false,
                'class' => null,
                'type' => 'array'
            ]
        ], $parameters);
    }

    /**
     * @test
     */
    public function getMethodParametersWithShortTypeNames()
    {
        $service = new ReflectionService();
        $parameters = $service->getMethodParameters(get_class($this), 'fixtureMethodForMethodTagsValuesWithShortTypes');
        $this->assertSame([
            'dummy' => [
                'position' => 0,
                'byReference' => false,
                'array' => false,
                'optional' => false,
                'allowsNull' => true,
                'class' => null,
                'type' => 'boolean'
            ],
            'foo' => [
                'position' => 1,
                'byReference' => false,
                'array' => false,
                'optional' => false,
                'allowsNull' => true,
                'class' => null,
                'type' => 'integer'
            ]
        ], $parameters);
    }

    /**
     * @test
     */
    public function reflectionServiceCanBeSerializedAndUnserialized()
    {
        $class = new class() {
        };

        $reflectionService = new ReflectionService();
        $serialized = serialize($reflectionService);
        unset($reflectionService);

        $configurationManagerProphecy = $this->prophesize(ConfigurationManagerInterface::class);
        $configurationManagerProphecy
            ->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FRAMEWORK)
            ->willReturn(['extensionName' => 'extkey'])
        ;

        $nullCacheFrontendProphecy = $this->prophesize(FrontendInterface::class);
        $nullCacheFrontendProphecy->has(Argument::any())->willReturn(false);
        $nullCacheFrontendProphecy->get(Argument::any())->willReturn(false);

        $cacheManagerProphecy = $this->prophesize(CacheManager::class);
        $cacheManagerProphecy
            ->getCache('extbase_reflection')
            ->willReturn($nullCacheFrontendProphecy)
        ;
        GeneralUtility::setSingletonInstance(CacheManager::class, $cacheManagerProphecy->reveal());

        $classSchema = $this->prophesize(ClassSchema::class)->reveal();

        $objectManagerMock = $this->prophesize(ObjectManager::class);
        $objectManagerMock->get(ConfigurationManagerInterface::class)->willReturn($configurationManagerProphecy->reveal());
        $objectManagerMock->get(ClassSchema::class, get_class($class))->willReturn($classSchema);

        GeneralUtility::setSingletonInstance(ObjectManager::class, $objectManagerMock->reveal());

        $reflectionService = unserialize($serialized, ['allowed_classes' => [ReflectionService::class]]);

        self::assertInstanceOf(ReflectionService::class, $reflectionService);
        self::assertInstanceOf(ClassSchema::class, $reflectionService->getClassSchema($class));
    }

    /**
     * @test
     */
    public function reflectionServiceIsResetDuringWakeUp()
    {
        $configurationManagerProphecy = $this->prophesize(ConfigurationManagerInterface::class);
        $configurationManagerProphecy
            ->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FRAMEWORK)
            ->willReturn(['extensionName' => 'extkey'])
        ;

        $nullCacheFrontendProphecy = $this->prophesize(FrontendInterface::class);
        $nullCacheFrontendProphecy->has(Argument::any())->willReturn(false);
        $nullCacheFrontendProphecy->get(Argument::any())->willReturn(false);

        $cacheManagerProphecy = $this->prophesize(CacheManager::class);
        $cacheManagerProphecy
            ->getCache('extbase_reflection')
            ->willReturn($nullCacheFrontendProphecy)
        ;
        GeneralUtility::setSingletonInstance(CacheManager::class, $cacheManagerProphecy->reveal());

        $objectManagerMock = $this->prophesize(ObjectManager::class);
        $objectManagerMock->get(ConfigurationManagerInterface::class)->willReturn($configurationManagerProphecy->reveal());

        GeneralUtility::setSingletonInstance(ObjectManager::class, $objectManagerMock->reveal());

        $insecureString = file_get_contents(__DIR__ . '/Fixture/InsecureSerializedReflectionService.txt');
        $reflectionService = unserialize($insecureString);

        $reflectionClass = new \ReflectionClass($reflectionService);
        $classSchemaProperty = $reflectionClass->getProperty('classSchemata');
        $classSchemaProperty->setAccessible(true);

        self::assertSame([], $classSchemaProperty->getValue($reflectionService));
    }

    protected function tearDown()
    {
        GeneralUtility::resetSingletonInstances($this->singletons);
        parent::tearDown();
    }
}
