<?php

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace TYPO3\CMS\Reports\Report\Status;

use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Localization\LanguageService;
use TYPO3\CMS\Core\Resource\Folder;
use TYPO3\CMS\Core\Resource\ResourceStorage;
use TYPO3\CMS\Core\Resource\Service\ResourceConsistencyService;
use TYPO3\CMS\Core\Resource\StorageRepository;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Validation\ResultException;
use TYPO3\CMS\Core\Validation\ResultRenderingTrait;
use TYPO3\CMS\Reports\Status as ReportStatus;
use TYPO3\CMS\Reports\StatusProviderInterface;

/**
 * Performs several checks about the FAL status
 */
class FalStatus implements StatusProviderInterface
{
    use ResultRenderingTrait;

    private ResourceConsistencyService $resourceConsistencyService;

    public function __construct()
    {
        $this->resourceConsistencyService = GeneralUtility::makeInstance(ResourceConsistencyService::class);
    }

    /**
     * Determines the status of the FAL index.
     *
     * @return array List of statuses
     */
    public function getStatus()
    {
        return [
            'MissingFiles' => $this->getMissingFilesStatus(),
            'ConsistencyCheck' => $this->getConsistencyCheckStatus(),
        ];
    }

    public function getDetailedStatus(): array
    {
        return [
            'ConsistencyCheck' => $this->getConsistencyCheckStatus(),
        ];
    }

    /**
     * Checks if there are files marked as missed.
     *
     * @return \TYPO3\CMS\Reports\Status An object representing whether there are files marked as missed or not
     */
    protected function getMissingFilesStatus()
    {
        $value = $this->getLanguageService()->getLL('status_none');
        $count = 0;
        $maxFilesToShow = 100;
        $message = '';
        $severity = ReportStatus::OK;

        $storageRepository = GeneralUtility::makeInstance(StorageRepository::class);
        $storageObjects = $storageRepository->findAll();
        $storages = [];

        /** @var ResourceStorage $storageObject */
        foreach ($storageObjects as $storageObject) {
            // We only check missing files for storages that are online
            if ($storageObject->isOnline()) {
                $storages[$storageObject->getUid()] = $storageObject;
            }
        }

        if (!empty($storages)) {
            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable('sys_file');
            $count = $queryBuilder
                ->count('*')
                ->from('sys_file')
                ->where(
                    $queryBuilder->expr()->eq(
                        'missing',
                        $queryBuilder->createNamedParameter(1, Connection::PARAM_INT)
                    ),
                    $queryBuilder->expr()->in(
                        'storage',
                        $queryBuilder->createNamedParameter(array_keys($storages), Connection::PARAM_INT_ARRAY)
                    )
                )
                ->executeQuery()
                ->fetchOne();
        }

        if ($count) {
            $value = sprintf($this->getLanguageService()->getLL('status_missingFilesCount'), $count);
            $severity = ReportStatus::WARNING;

            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable('sys_file');
            $files = $queryBuilder
                ->select('identifier', 'storage')
                ->from('sys_file')
                ->where(
                    $queryBuilder->expr()->eq(
                        'missing',
                        $queryBuilder->createNamedParameter(1, Connection::PARAM_INT)
                    ),
                    $queryBuilder->expr()->in(
                        'storage',
                        $queryBuilder->createNamedParameter(array_keys($storages), Connection::PARAM_INT_ARRAY)
                    )
                )
                ->setMaxResults($maxFilesToShow)
                ->executeQuery()
                ->fetchAllAssociative();

            $message = '<p>' . $this->getLanguageService()->getLL('status_missingFilesMessage') . '</p>';
            foreach ($files as $file) {
                $message .= $storages[$file['storage']]->getName() . ' ' . $file['identifier'] . '<br />';
            }

            if ($count > $maxFilesToShow) {
                $message .= '...<br />';
            }
        }

        return GeneralUtility::makeInstance(ReportStatus::class, $this->getLanguageService()->getLL('status_missingFiles'), $value, $message, $severity);
    }

    protected function getConsistencyCheckStatus(): ReportStatus
    {
        // @todo for performance reasons, consider using this only in CLI context as `ExtendedStatusProviderInterface`

        $storages = array_filter(
            GeneralUtility::makeInstance(StorageRepository::class)->findAll(),
            static fn(ResourceStorage $storage): bool => $storage->isOnline()
        );
        $inconsistenciesMessage = '';
        foreach ($storages as $storage) {
            $inconsistencies = $this->checkFolderConsistency($storage->getRootLevelFolder());
            if ($inconsistencies !== []) {
                $inconsistenciesMessage .= sprintf(
                    '<h5>%s</h5>%s',
                    htmlspecialchars(sprintf(
                        'Storage "%s" (id:%d)',
                        $storage->getName(),
                        $storage->getUid()
                    )),
                    $this->wrapInHtmlUnorderedList($inconsistencies)
                );
            }
        }
        if ($inconsistenciesMessage === '') {
            return GeneralUtility::makeInstance(
                ReportStatus::class,
                'Consistency check',
                'No inconsistencies found in these storages',
                $this->wrapInHtmlUnorderedList(array_map(
                    static fn(ResourceStorage $storage): string => sprintf(
                        '%s (id: %d)',
                        $storage->getName(),
                        $storage->getUid()
                    ),
                    $storages,
                )),
                ReportStatus::OK,
            );
        }
        return GeneralUtility::makeInstance(
            ReportStatus::class,
            'Consistency Status',
            'Inconsistent files have been found',
            $inconsistenciesMessage,
            ReportStatus::ERROR,
        );
    }

    private function checkFolderConsistency(Folder $folder): array
    {
        $inconsistencies = [];
        foreach ($folder->getFiles() as $file) {
            try {
                $this->resourceConsistencyService->validate($file->getStorage(), $file);
            } catch (ResultException $exception) {
                $inconsistencies[$file->getCombinedIdentifier()] = $this->compileResultMessages(
                    $exception->getMessages(),
                    $this->getLanguageService()
                );
            }
        }
        foreach ($folder->getSubFolders() as $subFolder) {
            $inconsistencies = array_merge($inconsistencies, $this->checkFolderConsistency($subFolder));
        }
        return $inconsistencies;
    }

    /**
     * @param list<string>|array<string, list<string>> $items
     */
    protected function wrapInHtmlUnorderedList(array $items): string
    {
        if (array_is_list($items)) {
            return sprintf(
                '<ul>%s</ul>',
                implode('', array_map(
                    static fn(string $item): string => '<li>' . htmlspecialchars($item) . '</li>',
                    $items
                ))
            );
        }
        return sprintf(
            '<ul>%s</ul>',
            implode('', array_map(
                fn(string $key, array $values): string => sprintf(
                    '<li>%s%s</li>',
                    htmlspecialchars($key),
                    $this->wrapInHtmlUnorderedList($values)
                ),
                array_keys($items),
                array_values($items)
            ))
        );
    }

    /**
     * @return LanguageService
     */
    protected function getLanguageService()
    {
        return $GLOBALS['LANG'];
    }
}
